/**
 * Perplexity Pro Activation - Main JavaScript
 * Handles UI interactions, API calls, and language switching
 */

// ========== Global State ==========
let currentLang = localStorage.getItem('preferredLang') || 'en';
let sessionId = '';
const uuid = document.getElementById('uuidData').value;

// ========== DOM Elements ==========
const languageModal = document.getElementById('languageModal');
const languageBtns = document.querySelectorAll('.language-btn');
const currentLangBtn = document.getElementById('currentLang');

const step1 = document.getElementById('step1');
const step2 = document.getElementById('step2');
const step3 = document.getElementById('step3');

const emailForm = document.getElementById('emailForm');
const emailInput = document.getElementById('emailInput');
const sendOtpBtn = document.getElementById('sendOtpBtn');
const emailMessage = document.getElementById('emailMessage');

const otpInputs = document.querySelectorAll('.otp-input');
const verifyOtpBtn = document.getElementById('verifyOtpBtn');
const otpMessage = document.getElementById('otpMessage');

// ========== Language Functions ==========
function setLanguage(lang) {
    currentLang = lang;
    localStorage.setItem('preferredLang', lang);

    // Update HTML lang attribute
    document.documentElement.lang = lang;

    // Apply RTL for Arabic and Kurdish
    if (lang === 'ar' || lang === 'ku') {
        document.body.classList.add('rtl');
        document.body.setAttribute('lang', lang);
    } else {
        document.body.classList.remove('rtl');
        document.body.removeAttribute('lang');
    }

    // Update all translatable elements
    const elements = document.querySelectorAll('[data-i18n]');
    elements.forEach(el => {
        const key = el.getAttribute('data-i18n');
        if (translations[lang] && translations[lang][key]) {
            el.textContent = translations[lang][key];
        }
    });

    // Update placeholders
    const placeholders = document.querySelectorAll('[data-i18n-placeholder]');
    placeholders.forEach(el => {
        const key = el.getAttribute('data-i18n-placeholder');
        if (translations[lang] && translations[lang][key]) {
            el.placeholder = translations[lang][key];
        }
    });

    // Update current language button
    const langNames = {
        en: 'English',
        zh: '中文',
        ar: 'العربية',
        ku: 'کوردی'
    };
    currentLangBtn.textContent = langNames[lang];
}

function translate(key) {
    return translations[currentLang]?.[key] || translations.en[key] || key;
}

// ========== Language Modal ==========
languageBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        const lang = btn.getAttribute('data-lang');
        setLanguage(lang);
        languageModal.classList.remove('active');
    });
});

currentLangBtn.addEventListener('click', () => {
    languageModal.classList.add('active');
});

// Show language modal on first visit (if no preference)
if (!localStorage.getItem('preferredLang')) {
    languageModal.classList.add('active');
} else {
    setLanguage(currentLang);
    languageModal.classList.remove('active');
}

// ========== OTP Input Handling ==========
otpInputs.forEach((input, index) => {
    // Auto-focus next input on type
    input.addEventListener('input', (e) => {
        const value = e.target.value;

        // Only allow digits
        if (!/^\d$/.test(value)) {
            e.target.value = '';
            return;
        }

        // Mark as filled
        if (value) {
            input.classList.add('filled');
            // Move to next input
            if (index < otpInputs.length - 1) {
                otpInputs[index + 1].focus();
            }
        } else {
            input.classList.remove('filled');
        }

        // Enable verify button if all inputs filled
        checkOtpComplete();
    });

    // Handle backspace
    input.addEventListener('keydown', (e) => {
        if (e.key === 'Backspace' && !input.value && index > 0) {
            otpInputs[index - 1].focus();
            otpInputs[index - 1].value = '';
            otpInputs[index - 1].classList.remove('filled');
            checkOtpComplete();
        }
    });

    // Handle paste
    input.addEventListener('paste', (e) => {
        e.preventDefault();
        const pastedData = e.clipboardData.getData('text').trim();
        const digits = pastedData.match(/\d/g);

        if (digits) {
            digits.slice(0, 6).forEach((digit, i) => {
                if (otpInputs[i]) {
                    otpInputs[i].value = digit;
                    otpInputs[i].classList.add('filled');
                }
            });
            checkOtpComplete();
            // Focus last input
            const lastIndex = Math.min(digits.length - 1, 5);
            otpInputs[lastIndex].focus();
        }
    });
});

function checkOtpComplete() {
    const allFilled = Array.from(otpInputs).every(input => input.value !== '');
    verifyOtpBtn.disabled = !allFilled;
}

function getOtpValue() {
    return Array.from(otpInputs).map(input => input.value).join('');
}

function clearOtp() {
    otpInputs.forEach(input => {
        input.value = '';
        input.classList.remove('filled');
    });
    verifyOtpBtn.disabled = true;
}

// ========== Message Functions ==========
function showMessage(element, message, type = 'info') {
    element.textContent = message;
    element.className = 'message show ' + type;
}

function hideMessage(element) {
    element.classList.remove('show');
}

// ========== API Functions ==========
async function sendOtp(email) {
    try {
        const response = await fetch('api/send-otp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ email, uuid })
        });

        const data = await response.json();
        return data;
    } catch (error) {
        console.error('Send OTP error:', error);
        return {
            success: false,
            error: 'network_error',
            message: translate('error-occurred')
        };
    }
}

async function verifyOtp(otp) {
    try {
        const response = await fetch('api/verify-otp.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ otp, uuid })
        });

        const data = await response.json();
        return data;
    } catch (error) {
        console.error('Verify OTP error:', error);
        return {
            success: false,
            error: 'network_error',
            message: translate('error-occurred')
        };
    }
}

// ========== Step 1: Email Submission ==========
emailForm.addEventListener('submit', async (e) => {
    e.preventDefault();

    const email = emailInput.value.trim();

    // Validate email
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
        showMessage(emailMessage, translate('invalid-email'), 'error');
        return;
    }

    // Disable button and show loading
    sendOtpBtn.disabled = true;
    sendOtpBtn.classList.add('loading');
    sendOtpBtn.querySelector('.btn-text').textContent = translate('sending');
    sendOtpBtn.querySelector('.spinner').style.display = 'inline-block';
    hideMessage(emailMessage);

    // Call API
    const result = await sendOtp(email);

    // Re-enable button
    sendOtpBtn.disabled = false;
    sendOtpBtn.classList.remove('loading');
    sendOtpBtn.querySelector('.btn-text').textContent = translate('send-code');
    sendOtpBtn.querySelector('.spinner').style.display = 'none';

    if (result.success) {
        sessionId = result.sessionId;
        showMessage(emailMessage, translate('code-sent'), 'success');

        // Move to step 2 after short delay
        setTimeout(() => {
            step1.classList.add('hidden');
            step2.classList.remove('hidden');
            otpInputs[0].focus();
        }, 1500);
    } else {
        // Show error message
        let errorMsg = result.message || translate('send-failed');

        // Use translated error messages
        if (result.error === 'already_used') {
            errorMsg = translate('already-used');
        } else if (result.error === 'invalid_code') {
            errorMsg = translate('invalid-link');
        } else if (result.error === 'too_many_attempts') {
            errorMsg = translate('too-many-attempts');
        }

        showMessage(emailMessage, errorMsg, 'error');
    }
});

// ========== Step 2: OTP Verification ==========
verifyOtpBtn.addEventListener('click', async () => {
    const otp = getOtpValue();

    if (otp.length !== 6) {
        showMessage(otpMessage, translate('invalid-otp'), 'error');
        return;
    }

    // Disable button and show loading
    verifyOtpBtn.disabled = true;
    verifyOtpBtn.classList.add('loading');
    verifyOtpBtn.querySelector('.btn-text').textContent = translate('verifying');
    verifyOtpBtn.querySelector('.spinner').style.display = 'inline-block';
    hideMessage(otpMessage);

    // Call API
    const result = await verifyOtp(otp);

    if (result.success) {
        // Success! Move to step 3
        showMessage(otpMessage, translate('success'), 'success');

        setTimeout(() => {
            step2.classList.add('hidden');
            step3.classList.remove('hidden');
        }, 1000);
    } else {
        // Re-enable button
        verifyOtpBtn.disabled = false;
        verifyOtpBtn.classList.remove('loading');
        verifyOtpBtn.querySelector('.btn-text').textContent = translate('verify');
        verifyOtpBtn.querySelector('.spinner').style.display = 'none';

        // Show error message
        let errorMsg = result.message || translate('verification-failed');

        // Use translated error messages
        if (result.error === 'invalid_otp') {
            errorMsg = translate('invalid-otp');
        } else if (result.error === 'already_subscribed') {
            errorMsg = translate('already-subscribed');
        } else if (result.error === 'promo_invalid') {
            errorMsg = translate('promo-invalid');
        } else if (result.error === 'session_expired' || result.error === 'session_not_found') {
            errorMsg = translate('session-expired');
        }

        showMessage(otpMessage, errorMsg, 'error');

        // Clear OTP inputs for retry
        clearOtp();
        otpInputs[0].focus();
    }
});

// ========== Keyboard Shortcuts ==========
document.addEventListener('keydown', (e) => {
    // Press Enter on OTP inputs to verify
    if (document.activeElement.classList.contains('otp-input') && e.key === 'Enter') {
        if (!verifyOtpBtn.disabled) {
            verifyOtpBtn.click();
        }
    }
});

// ========== Initialize ==========
// Apply initial language
setLanguage(currentLang);

// Focus email input if step 1 is visible
if (!step1.classList.contains('hidden')) {
    emailInput.focus();
}

console.log('Perplexity Pro Activation System loaded');
